﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Net;
using System.Linq;
using System.Security;
using System.ServiceModel;
using System.Text;
using VA.PPMS.Context;
using VA.PPMS.CRM.Plugins.Helper;
using System.Web;

namespace VA.PPMS.CRM.Plugins
{
    public class CareSiteCreate : IPlugin
    {
        private const string PluginName = "CareSiteCreate";
        private IOrganizationService _service;
        private ITracingService _tracingService;

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity entity = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (entity.LogicalName != "ppms_caresite")
                    return;

                _tracingService.Trace("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                // Verify service reference exists
                if (_service == null)
                {
                    _tracingService.Trace("Service is unavailable.");
                    return;
                }

                //Initiate the Address Validation Data. 
                var addressValidationData = new RootObjectResponse();
                addressValidationData = null;

                try
                {
                    using (var svc = new PpmsContext(_service))
                    {
                        _tracingService.Trace("Getting address attributes for care site...");
                        // Get Care Site Address Attributes
                        var careSite = svc.ppms_caresiteSet.First(i => i.Id == entity.Id);
                        svc.LoadProperty(careSite, "ppms_organization");

                        _tracingService.Trace("Processing Care Site Address: {0}", careSite.ppms_name);

                        // Check to see if address is provided
                        if (careSite != null)
                        {
                            string streetAddress = careSite.ppms_address_line1;
                            string city = careSite.ppms_address_city;
                            string state = careSite.ppms_statename;
                            string zip = careSite.ppms_address_postalcode;
                            string address = streetAddress + " " + city + " " + state + " " + zip;

                            //Check the Use Address Validation? Field
                            bool useAddressValidation = (bool)careSite.Attributes["ppms_useaddressvalidationservice"];

                            if (useAddressValidation == true)
                            {
                                _tracingService.Trace("Address Validation Service");
                                //Address & Geocode Validation
                                addressValidationData = AddressValidationHelper.ValidateWcIWS(careSite, streetAddress, city, state, zip, _tracingService, svc, _service);

                                if (addressValidationData != null)
                                {
                                    if (addressValidationData.addressMetaData.confidenceScore >= 70)
                                    {
                                        _tracingService.Trace("Address Validation successful");
                                        // Set geocode values                                  
                                        var coordinates = new PpmsCoordinate()
                                        {
                                            ConfidenceScore = addressValidationData.addressMetaData.confidenceScore,
                                            Latitude = (decimal)addressValidationData.geocode.latitude,
                                            Longitude = (decimal)addressValidationData.geocode.longitude
                                        };
                                        SetCoordinates(entity, coordinates);
                                        SetAddressData(entity, addressValidationData);
                                        _tracingService.Trace("Set address and geocode validated...");
                                        SetAddressValidated(entity);
                                        // Create provider validation entities
                                        _tracingService.Trace("Create provider validation entries...");
                                        SetProviderValidation(entity);
                                        ActivateRelations(entity);
                                    }
                                    else
                                    {
                                        //Regardless of Passing our criteria put the Address Validation feedback on form. 
                                        SetAddressData(entity, addressValidationData);
                                        //Deactivate Relations                                       
                                        DeactivateRelations(entity);
                                    }
                                }

                                //Set Provider Validation for Parent Provider related to Care Site
                                _tracingService.Trace("Set address and geocode validated...");
                                UpdateParentProviderValidation(careSite);

                            }
                        }
                    }
                    return;
                }
                catch (WebException we)
                {
                    _tracingService.Trace("Web Exception On Call to Address Validation Service", we.ToString());
                    DeactivateRelations(entity);
                    return;
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    return;
                }
            }
            _tracingService.Trace("Done");
        }

        private void SetCoordinates(Entity entity, PpmsCoordinate coordinates)
        {
            // check if change needs to be made
            if (entity != null)
            {
                var updateCareSite = new ppms_caresite
                {
                    Id = entity.Id
                };

                // set latitude
                updateCareSite.ppms_latitude = coordinates.Latitude;
                // set longitude
                updateCareSite.ppms_longitude = coordinates.Longitude;
                // save changes
                _service.Update(updateCareSite);
            }
        }

        private void SetAddressValidated(Entity entity)
        {
            // check if change needs to be made
            if (entity != null)
            {
                var updateCareSite = new ppms_caresite
                {
                    Id = entity.Id
                };

                //Set Entity Geocode and Address Validated fields to True
                updateCareSite.Attributes["ppms_geocoded"] = true;
                updateCareSite.Attributes["ppms_addressvalidated"] = true;

                // save changes
                _service.Update(updateCareSite);
            }
        }

        private void SetAddressData(Entity entity, RootObjectResponse addressData)
        {
            // check if change needs to be made
            if (entity != null)
            {
                var updateCareSite = new ppms_caresite
                {
                    Id = entity.Id
                };

                //Set Entity Geocode and Address Validated fields to True
                updateCareSite.Attributes["ppms_addressvalidationconfidencescore"] = addressData.addressMetaData.confidenceScore;

                //Encode the Delivery Point Validation Text to ensure no Script can be rendered. 
                var deliveryPointValidTextEncoded = WebUtility.HtmlEncode(addressData.addressMetaData.deliveryPointValidation);
                updateCareSite.Attributes["ppms_deliverypointvalidation"] = deliveryPointValidTextEncoded;

                // save changes
                _service.Update(updateCareSite);
            }
        }

        private void DeactivateRelations(Entity entity)
        {         
            // Deactivate owning group
            var org = entity.GetAttributeValue<EntityReference>("ppms_organization");
            if (org != null)
            {
                _tracingService.Trace("Deactivate organization");
                var provider = GetProvider(org.Id);
                if (provider != null)
                {
                    _service.Execute(PpmsHelper.GetDeactivateRequest(provider,
                        (int)PpmsHelper.Account_StatusCode.AddressValidation));
                }
            }

            // Deactivate provider services
            var list = GetActiveServicesByCareSite(entity.Id);
            if (list != null)
            {
                _tracingService.Trace("Deactivate Provider Services");
                foreach (var item in list.Entities)
                {
                    _service.Execute(PpmsHelper.GetDeactivateRequest(item,
                        (int)PpmsHelper.ProviderService_StatusCode.AddressValidationFailure));
                }
            }

            _tracingService.Trace("Deactivate Care Site");
            //Address Validation unsuccessful, set state code
            _service.Execute(PpmsHelper.GetDeactivateRequest(entity,
                (int)PpmsHelper.CareSite_StatusCode.AddressValidationFailure));
        }

        private void ActivateRelations(Entity entity)
        {

            _tracingService.Trace("Activate Relations");

            //Address Validation successful, set state code
            _service.Execute(PpmsHelper.GetActivateRequest(entity,
                (int)PpmsHelper.CareSite_StatusCode.Active));

            // Activate owning group
            var org = entity.GetAttributeValue<EntityReference>("ppms_organization");
            if (org != null)
            {
                _tracingService.Trace("Activate organization");
                var provider = GetProvider(org.Id);
                if (provider != null)
                {
                    _service.Execute(PpmsHelper.GetActivateRequest(provider,
                        (int)PpmsHelper.Account_StatusCode.Active));
                }
            }

            //Reactivate provider services
            var list = GetInActiveServicesByCareSite(entity.Id);
            if (list != null)
            {
                _tracingService.Trace("Activate provider services");
                foreach (var item in list.Entities)
                {
                    _service.Execute(PpmsHelper.GetActivateRequest(item,
                        (int)PpmsHelper.ProviderService_StatusCode.Active));
                }
            }
        }


        /// <summary>
        /// Validate care site existence for type 2 providers
        /// </summary>
        /// <param name="careSite">New care site entity</param>
        private void UpdateParentProviderValidation(ppms_caresite careSite)
        {
            using (var svc = new PpmsContext(_service))
            {
                if (careSite != null && careSite.ppms_organization != null)
                {
                    _tracingService.Trace("Update Parent Provider");
                    var provider = svc.AccountSet.FirstOrDefault(i => i.Id == careSite.ppms_organization.Id);
                    if (provider != null)
                    {
                        _tracingService.Trace("--Related provider found. ({0})", provider.ppms_providertype.Value);
                        if (provider.ppms_providertype.Value == (int)ppms_ProviderType.GroupPracticeAgency)
                        {
                            _tracingService.Trace("---Processing Type 2 provider");
                            var historyLog = new ppms_historylog()
                            {
                                ppms_name = "Validation Update: Care Site",
                                ppms_providerid = provider.ToEntityReference(),
                                ppms_type = new OptionSetValue((int)ppms_historylog_ppms_type.ValidationCareSite),
                                ppms_validated = true
                            };

                            _tracingService.Trace("---Saving history log");
                            //save changes
                            _service.Create(historyLog);
                        }
                    }
                }
            }
        }


        private Entity GetProvider(Guid providerId)
        {
            return _service.Retrieve("account", providerId, new ColumnSet(new string[] { "accountid", "name" }));
        }

        private EntityCollection GetActiveServicesByCareSite(Guid careSiteId)
        {
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("statecode", ConditionOperator.Equal, (int)PpmsHelper.AccountState.Active);
            filter.AddCondition("ppms_caresite", ConditionOperator.Equal, careSiteId);

            QueryExpression query = new QueryExpression("ppms_providerservice");
            query.ColumnSet.AddColumns("ppms_providerserviceid", "ppms_name", "ppms_providerid");
            query.Criteria.AddFilter(filter);

            return _service.RetrieveMultiple(query);
        }

        private EntityCollection GetInActiveServicesByCareSite(Guid careSiteId)
        {
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("statecode", ConditionOperator.Equal, (int)PpmsHelper.AccountState.Inactive);
            filter.AddCondition("ppms_caresite", ConditionOperator.Equal, careSiteId);

            QueryExpression query = new QueryExpression("ppms_providerservice");
            query.ColumnSet.AddColumns("ppms_providerserviceid", "ppms_name", "ppms_providerid");
            query.Criteria.AddFilter(filter);

            return _service.RetrieveMultiple(query);
        }

        /*
        private void AddAddressString(StringBuilder sb, Entity entity, string propertyName)
        {
            var element = entity.GetAttributeValue<string>(propertyName);
            if (!string.IsNullOrEmpty(element))
            {
                if (sb.Length > 0) sb.Append(",");
                sb.Append(element);
            }
        }
        

        private void AddAddressEntity(StringBuilder sb, Entity entity, string propertyName)
        {
            var element = entity.GetAttributeValue<EntityReference>(propertyName);
            if (element != null && !string.IsNullOrEmpty(element.Name))
            {
                if (sb.Length > 0) sb.Append(",");
                sb.Append(element.Name);
            }
        }
        */

        private void SetProviderValidation(Entity entity)
        {
            // Get associated provider services
            var services = GetActiveServicesByCareSite(entity.Id);
            if (services != null && services.Entities.Count > 0)
            {
                foreach (var service in services.Entities)
                {
                    // Create provider validations
                    AddProviderValidations(service.GetAttributeValue<EntityReference>("ppms_providerid"));
                }
            }
        }

        private bool AddProviderValidations(EntityReference provider)
        {
            if (provider != null)
            {
                // Add Address validation
                PpmsHelper.AddProviderValidation(_service, provider, PpmsHelper.Validation_Type.Address);

                // Add Geocode validation
                PpmsHelper.AddProviderValidation(_service, provider, PpmsHelper.Validation_Type.Geocode);

                return true;
            }

            return false;
        }
    }
}
